// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Cross section for e+e- K0SK+-\pi+-\pi0 and K0SK+-pi-+eta between 3.90 to 4.60 GeV
  class BESIII_2018_I1699641 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2018_I1699641);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");

      // Book histograms
      for (size_t ix=0; ix<2; ++ix) {
        book(_sigma[ix], 1+ix, 1, 1);
      }

      for (const string& en : _sigma[0].binning().edges<0>()) {
        const double eval = stod(en)*GeV;
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      if (_sqs.empty() && isCompatibleWithSqrtS(4.085))  _sqs = "4.085"s;
      else raiseBeamErrorIf(_sqs.empty());
    }


    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // K K pi pi
      if (ntotal==4 && nCount[310]==1 && nCount[111]==1 &&
        ((nCount[ 321]==1 &&nCount[-211]==1) ||
         (nCount[-321]==1 &&nCount[ 211]==1) )) {
        _sigma[0]->fill(_sqs);
      }
      // eta resonance
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      for (const Particle& p : ufs.particles()) {
        if (p.children().empty()) continue;
        if (p.pid()!=221) continue;
        map<long,int> nRes=nCount;
        int ncount = ntotal;
        findChildren(p,nRes,ncount);
        if (ncount!=3) continue;
        bool matched=true;
        for (const auto& val : nRes) {
          if (abs(val.first)==321 || abs(val.first)==211) {
            continue;
          }
          else if (abs(val.first)==310) {
            if (val.second!=1) {
              matched = false;
              break;
            }
          }
          else if (val.second!=0) {
            matched = false;
            break;
          }
        }
        if (matched==false) continue;
        if ((nCount[ 321] == 1 && nCount[-211] ==1) ||
            (nCount[-321] == 1 && nCount[ 211] ==1)) {
          _sigma[1]->fill(_sqs);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma, crossSection()/ sumOfWeights() /picobarn);
    }
    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigma[2];
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(BESIII_2018_I1699641);


}
